<?php

namespace Inspire_Labs\Empik_Woocommerce\Product;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use WC_Product_CSV_Exporter;

/**
 * Include dependencies.
 */
if ( ! class_exists( 'WC_CSV_Batch_Exporter', false ) ) {
	include_once WC_ABSPATH . 'includes/export/abstract-wc-csv-batch-exporter.php';
}


/**
 * Empik_WC_Product_Exporter
 */
class Empik_WC_Product_Exporter extends WC_Product_CSV_Exporter {


	/**
	 * Constructor.
	 */
	public function __construct() {
		parent::__construct();
	}


	/**
	 * Return an array of columns to export.
	 *
	 * @since  3.1.0
	 * @return array
	 */
	public function get_default_column_names() {
		return apply_filters(
			"woocommerce_product_export_{$this->export_type}_default_columns",
			$this->get_standard_woo_export_columns()
		);
	}


	/**
	 * Prepare data for export.
	 *
	 * @since 3.1.0
	 */
	public function prepare_data_to_export() {

		ini_set( 'max_execution_time', '28800' );

		$found_attributes = $this->get_all_product_attributes( 'product' );

		$args = array(
			'status'   => array( 'private', 'publish', 'draft', 'future', 'pending' ),
			'type'     => $this->product_types_to_export,
			'limit'    => $this->get_limit(),
			'page'     => $this->get_page(),
			'orderby'  => array(
				'ID' => 'ASC',
			),
			'return'   => 'objects',
			'paginate' => true,
		);

		if ( ! empty( $this->product_category_to_export ) ) {
			$args['category'] = $this->product_category_to_export;
		}
		$products = wc_get_products( apply_filters( "woocommerce_product_export_{$this->export_type}_query_args", $args ) );

		$this->total_rows = $products->total;
		$this->row_data   = array();

		foreach ( $products->products as $product ) {
			$this->row_data[] = $this->generate_row_data_empik( $product, $found_attributes );
		}
	}

	/**
	 * Generates row data for Empik product export.
	 *
	 * Creates a complete export row for a product by processing all column values,
	 * handling special cases like EAN codes, tax values, and category inheritance
	 * for variations. Applies formatting and validation for Empik API compatibility.
	 *
	 * @param \WC_Product $product          The WooCommerce product object to export.
	 * @param array       $found_attributes Array of product attributes found during processing.
	 * @return array Associative array of column data for the product export row.
	 */
	protected function generate_row_data_empik( $product, $found_attributes ) {

		$columns = $this->get_column_names();

        $product_id = $product->get_id();

        $is_variation = false;
        if ( $product->is_type( 'variation' ) ) {
            $is_variation = true;
        }

		$row = array();

		foreach ( $columns as $column_id => $column_name ) {

			$column_id = strstr( $column_id, ':' ) ? current( explode( ':', $column_id ) ) : $column_id;
			$value     = '';

			if ( 'code_ean' === $column_id ) {
				$value      = '';
				$meta_field = get_option( 'empik_woocommerce_set_export_offers_id_field' );
				if ( ! empty( $meta_field ) ) {
					$ean_value  = get_post_meta( $product_id, $meta_field, true );
					if ( ! empty( $ean_value ) ) {
						$value = $ean_value;
					}
				}
			}

			if ( 'tax_value' === $column_id ) {
				$value = '';
				$tax   = new \WC_Tax();
				if ( is_object( $tax ) && ! is_wp_error( $tax ) ) {
					$tax_rates_data = $tax->find_rates(
						array(
							'country'   => 'PL',
							'state'     => '',
							'city'      => '',
							'postcode'  => '',
							'tax_class' => $product->get_tax_class(),
						)
					);

					if ( ! empty( $tax_rates_data ) ) {
						$value = reset( $tax_rates_data )['rate'];
					}
				}
			}

			// Skip some columns if dynamically handled later or if we're being selective.
			if ( in_array( $column_id, array( 'downloads', 'attributes', 'meta' ), true ) || ! $this->is_column_exporting( $column_id ) ) {
				continue;
			}

			if ( has_filter( "woocommerce_product_export_{$this->export_type}_column_{$column_id}" ) ) {
				// Filter for 3rd parties.
				$value = apply_filters( "woocommerce_product_export_{$this->export_type}_column_{$column_id}", '', $product, $column_id );

			} elseif ( is_callable( array( $this, "get_column_value_{$column_id}" ) ) ) {
				// Handle special columns which don't map 1:1 to product data.
				$value = $this->{"get_column_value_{$column_id}"}( $product );

				// add parent category for variations.
				if ( 'category_ids' === $column_id ) {
					if ( $is_variation ) {
						$value  = '';
						$parent = wc_get_product( $product->get_parent_id() );
						if ( $parent && is_object( $parent ) && ! is_wp_error( $parent ) ) {
							$term_ids      = $parent->get_category_ids( 'edit' );
							$variation_cat = $this->format_term_ids( $term_ids, 'product_cat' );
							$value         = $variation_cat;
						}
					}
				}
			} elseif ( is_callable( array( $product, "get_{$column_id}" ) ) ) {
				// Default and custom handling.
				$value = $product->{"get_{$column_id}"}( 'edit' );
			}

			if ( 'description' === $column_id || 'short_description' === $column_id ) {
				if ( defined( 'EMPIK_TEST_MODE_ON' ) && true === EMPIK_TEST_MODE_ON ) {
					$value = ''; // for easier analyze of test .csv export file.
				} else {
                    if( $is_variation && empty( $value ) ) {
                        $parent_id = $product->get_parent_id();
                        $parent_product = wc_get_product( $parent_id );
                        if( $parent_product && ! is_wp_error( $parent_product ) ) {
                            if ('description' === $column_id && empty( $value ) ) {
                                $value = $parent_product->get_description();
                            }
                            if ( 'short_description' === $column_id && empty( $value ) ) {
                                $value = $parent_product->get_short_description();
                            }
                        }
                    }
					$value = $this->filter_description_field( $value );
				}
			}

            if ( 'images' === $column_id  ) {
                if( $is_variation && empty( $value ) ) {
                    $parent_id = $product->get_parent_id();
                    $parent_product = wc_get_product( $parent_id );
                    if( $parent_product && ! is_wp_error( $parent_product ) ) {
                        $parent_feature_image_id = get_post_thumbnail_id( $parent_product->get_id() );
                        $value = wp_get_attachment_image_url( $parent_feature_image_id, 'full' );
                    }
                }
            }


			// comma in prices cause the data structure is broken, so we need validate it.
			if ( $value !== null ) {
				if ( strpos( $value, ',' ) !== false ) {
					$value = $value . ' ';
				}
			}

            if ( 'name' === $column_id ) {
                // if product Variation - add name of first attribute to product name.
                if( $is_variation ) {
                    $first_attr_value = $this->get_variation_attribute_value( $product_id );
                    if( ! empty( $first_attr_value ) ) {
                        $value .= ', ' . $first_attr_value;
                    }
                }
            }

			$row[ $column_id ] = $value;

		}

		$this->prepare_downloads_for_export( $product, $row );
		// changed for attributes.
        $this->prepare_attributes_for_export_empik( $found_attributes, $product, $row );
		$this->prepare_meta_for_export( $product, $row );

		return apply_filters( 'woocommerce_product_export_row_data', $row, $product );
	}



	/**
	 * Prepares product attributes for Empik export format.
	 *
	 * Processes product attributes and adds them to the export row with unique
	 * column names. Handles both taxonomy and custom attributes, ensuring no
	 * duplicate column names and proper value formatting for the Empik API.
	 *
	 * @param array       $all_attributes Array of all available attribute keys.
	 * @param \WC_Product $product        The WooCommerce product object.
	 * @param array       &$row           Reference to the export row array to populate.
	 */
	protected function prepare_attributes_for_export_empik( $all_attributes, $product, &$row ) {


        $leave_attrbute_data_blank = false;
        $product_id = $product->get_id();

        if( $product->get_type() === 'variable'  ) {
            $leave_attrbute_data_blank = true;
        }

		if ( $this->is_column_exporting( 'attributes' ) ) {
			$attributes = $product->get_attributes();

			if ( count( $attributes ) ) {

				$uniq_columns = array();

				$i = 1;
				foreach ( $all_attributes as $fa ) {
					$column_name = wc_attribute_label( $fa, $product );

					if ( empty( $column_name ) ) {
						continue;
					}

					// API doesn't accept files with duplicate column names, so we need make it unique.
					if ( ! in_array( $column_name, $uniq_columns ) ) {
						$uniq_columns[] = $column_name; // only unique column names.
					} else {
						// $column_name = $column_name . '_' . $i;
						continue;
					}

					// same validation for duplication column names in case it matches any Woo.
					// standard export field name
					if ( in_array( $column_name, $this->get_standard_woo_export_columns() ) ) {
						$column_name = $column_name . '_' . $i;
					}

					$this->column_names[ 'attributes:name' . $i ] = $column_name;

					foreach ( $attributes as $attribute_name => $attribute ) {

                        if( $product->is_type('variation') && is_string( $attribute ) ) {
                            if ( strtolower( $fa ) === $attribute_name ) {

                                $value = $this->get_variation_attribute_value( $product_id, $attribute_name );

                                $value =  str_replace(',', '\\,', $value );

                                $row['attributes:name' . $i] = $value;

                                if ($leave_attrbute_data_blank) {
                                    $row['attributes:name' . $i] = '';
                                }
                            }

                        } else
						if ( is_object( $attribute ) ) {


							$attribute_data = $attribute->get_data();

							if ( strtolower( $fa ) === $attribute_name || $fa == $attribute_data['name'] ) {

								if ( is_a( $attribute, 'WC_Product_Attribute' ) ) {

									if ( $attribute->is_taxonomy() ) {

										$terms  = $attribute->get_terms();
										$values = array();

										foreach ( $terms as $term ) {
											$values[] = $term->name;
										}

										$row[ 'attributes:name' . $i ] = $this->implode_values( $values );
                                        if( $leave_attrbute_data_blank ) {
                                            $row[ 'attributes:name' . $i ] = '';
                                        }

									} else {

										$row[ 'attributes:name' . $i ] = $this->implode_values( $attribute->get_options() );
                                        if( $leave_attrbute_data_blank ) {
                                            $row[ 'attributes:name' . $i ] = '';
                                        }

									}
								} elseif ( 0 === strpos( $attribute_name, 'pa_' ) ) {

                                    $option_term = get_term_by('slug', $attribute, $attribute_name); // @codingStandardsIgnoreLine.
                                    $row[ 'attributes:name' . $i ] = $option_term && ! is_wp_error( $option_term ) ? str_replace( ',', '\\,', $option_term->name ) : str_replace( ',', '\\,', $attribute );
                                    if( $leave_attrbute_data_blank ) {
                                        $row[ 'attributes:name' . $i ] = '';
                                    }

								} else {

									$row[ 'attributes:name' . $i ] = str_replace( ',', '\\,', $attribute );
                                    if( $leave_attrbute_data_blank ) {
                                        $row[ 'attributes:name' . $i ] = '';
                                    }
								}
							}
						}
					}
					++$i;
				}
			}
		}
	}


	/**
	 * Returns the standard WooCommerce export column definitions.
	 *
	 * Provides an associative array mapping column keys to their translated
	 * display names for product export functionality. Includes all standard
	 * WooCommerce product fields plus custom EAN code field.
	 *
	 * @return array Associative array of column keys and their translated labels.
	 */
	protected function get_standard_woo_export_columns() {
		return array(
			'id'                 => esc_html__( 'ID', 'woocommerce' ),
			'type'               => esc_html__( 'Type', 'woocommerce' ),
			'sku'                => esc_html__( 'SKU', 'woocommerce' ),
			'name'               => esc_html__( 'Name', 'woocommerce' ),
			'published'          => esc_html__( 'Published', 'woocommerce' ),
			'featured'           => esc_html__( 'Is featured?', 'woocommerce' ),
			'catalog_visibility' => esc_html__( 'Visibility in catalog', 'woocommerce' ),
			'short_description'  => esc_html__( 'Short description', 'woocommerce' ),
			'description'        => esc_html__( 'Description', 'woocommerce' ),
			'date_on_sale_from'  => esc_html__( 'Date sale price starts', 'woocommerce' ),
			'date_on_sale_to'    => esc_html__( 'Date sale price ends', 'woocommerce' ),
			'tax_status'         => esc_html__( 'Tax status', 'woocommerce' ),
			'tax_class'          => esc_html__( 'Tax class', 'woocommerce' ),
			'tax_value'          => esc_html__( 'VAT', 'woocommerce' ),
			'stock_status'       => esc_html__( 'In stock?', 'woocommerce' ),
			'stock'              => esc_html__( 'Stock', 'woocommerce' ),
			'low_stock_amount'   => esc_html__( 'Low stock amount', 'woocommerce' ),
			'backorders'         => esc_html__( 'Backorders allowed?', 'woocommerce' ),
			'sold_individually'  => esc_html__( 'Sold individually?', 'woocommerce' ),
			/* translators: %s: weight */
			'weight'             => sprintf( esc_html__( 'Weight (%s)', 'woocommerce' ), get_option( 'woocommerce_weight_unit' ) ),
			/* translators: %s: length */
			'length'             => sprintf( esc_html__( 'Length (%s)', 'woocommerce' ), get_option( 'woocommerce_dimension_unit' ) ),
			/* translators: %s: width */
			'width'              => sprintf( esc_html__( 'Width (%s)', 'woocommerce' ), get_option( 'woocommerce_dimension_unit' ) ),
			/* translators: %s: Height */
			'height'             => sprintf( esc_html__( 'Height (%s)', 'woocommerce' ), get_option( 'woocommerce_dimension_unit' ) ),
			'reviews_allowed'    => esc_html__( 'Allow customer reviews?', 'woocommerce' ),
			'purchase_note'      => esc_html__( 'Purchase note', 'woocommerce' ),
			'sale_price'         => esc_html__( 'Sale price', 'woocommerce' ),
			'regular_price'      => esc_html__( 'Regular price', 'woocommerce' ),
			'category_ids'       => esc_html__( 'Categories', 'woocommerce' ),
			'tag_ids'            => esc_html__( 'Tags', 'woocommerce' ),
			'shipping_class_id'  => esc_html__( 'Shipping class', 'woocommerce' ),
			'images'             => esc_html__( 'Images', 'woocommerce' ),
			'download_limit'     => esc_html__( 'Download limit', 'woocommerce' ),
			'download_expiry'    => esc_html__( 'Download expiry days', 'woocommerce' ),
			'parent_id'          => esc_html__( 'Parent', 'woocommerce' ),
			'grouped_products'   => esc_html__( 'Grouped products', 'woocommerce' ),
			'upsell_ids'         => esc_html__( 'Upsells', 'woocommerce' ),
			'cross_sell_ids'     => esc_html__( 'Cross-sells', 'woocommerce' ),
			'product_url'        => esc_html__( 'External URL', 'woocommerce' ),
			'button_text'        => esc_html__( 'Button text', 'woocommerce' ),
			'menu_order'         => esc_html__( 'Position', 'woocommerce' ),
			'code_ean'           => esc_html__( 'Code EAN', 'empik-for-woocommerce' ),
		);
	}


	/**
	 * Retrieves all unique product attributes from the database.
	 *
	 * Queries the database for all product attribute meta values, unserializes them,
	 * and extracts unique attribute keys. Returns a sorted array of attribute names
	 * or taxonomy slugs.
	 *
	 * @param string $post_type The post type to query for attributes. Default 'product'.
	 * @return array Sorted array of unique attribute keys/names.
	 */
	public function get_all_product_attributes( $post_type = 'product' ) {
		global $wpdb;

		$results = $wpdb->get_col(
			$wpdb->prepare(
				"SELECT DISTINCT pm.meta_value
            FROM {$wpdb->postmeta} AS pm
            LEFT JOIN {$wpdb->posts} AS p ON p.ID = pm.post_id
            WHERE p.post_type = %s
            AND p.post_status IN ( 'publish', 'pending', 'private', 'draft' )
            AND pm.meta_key = '_product_attributes'",
				$post_type
			)
		);

		// Go through each result, and look at the attribute keys within them.
		$result = array();

		if ( ! empty( $results ) ) {
			foreach ( $results as $_product_attributes ) {
				$attributes = maybe_unserialize( maybe_unserialize( $_product_attributes ) );
				if ( ! empty( $attributes ) && is_array( $attributes ) ) {
					foreach ( $attributes as $key => $attribute ) {
						if ( ! $key ) {
							continue;
						}
						if ( ! strstr( $key, 'pa_' ) ) {
							if ( empty( $attribute['name'] ) ) {
								continue;
							}
							$key = $attribute['name'];
						}

						$result[ $key ] = $key;
					}
				}
			}
		}

		sort( $result );

		return $result;
	}

    /**
     * Gets the attribute value for a product variation.
     *
     * @param int    $variation_id   The ID of the variation product.
     * @param string $attribute_name The attribute name to retrieve (e.g. 'pa_color'). If false, returns the first attribute value.
     *
     * @return string The attribute value or empty string if not found.
     */
    private function get_variation_attribute_value( $variation_id, $attribute_name = false ) {

        $product_variation = wc_get_product( $variation_id );

        if ( ! $product_variation || ! $product_variation->is_type('variation') ) {
            return '';
        }

        // Get all attributes of the variation.
        $attributes = $product_variation->get_attributes();

        if( ! $attribute_name ) {
            if ( ! empty($attributes) && is_array($attributes)) {
                $first_attribute = array_key_first($attributes);
                $attribute_value = $product_variation->get_attribute($first_attribute);
                if ( ! empty($attribute_value) ) {
                    return $attribute_value;
                }
            }
        }

        $attribute_value = $product_variation->get_attribute($attribute_name);
        if (!empty($attribute_value)) {
            return $attribute_value;
        }

        return '';
    }



}
