<?php

namespace Inspire_Labs\Empik_Woocommerce\Product;

if ( ! defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

use Exception;
use Inspire_Labs\Empik_Woocommerce\Api_Client\Ajax_Response_Model;
use Inspire_Labs\Empik_Woocommerce\Offer\Offers_Exporter;
use Inspire_Labs\Empik_Woocommerce\Plugin;
use Inspire_Labs\Empik_Woocommerce\Wp_Admin\Abstract_Exporter;
use Inspire_Labs\Empik_Woocommerce\Wp_Admin\Products_List_Mod_Factory;
use Inspire_Labs\Empik_Woocommerce\Wp_Admin\Products_List_Product_import_Column;
use Inspire_Labs\Empik_Woocommerce\Product\Empik_WC_Product_Exporter;
use Inspire_Labs\Empik_Woocommerce\Wp_Admin\Settings_Ids;
use WC_CSV_Exporter;
use WC_Product;
use WC_Product_CSV_Exporter;
use WP_Query;

/**
 * Product_Exporter
 */
class Product_Exporter extends Abstract_Exporter
{

    private static $full_columns_headers;


    protected function do_after_export_action(
        Ajax_Response_Model $ajax_response_model
    ) {
        return;
    }

    /**
     * Returns the exporter type identifier.
     *
     * @return int The exporter type constant for products.
     */
    protected function get_type(): int {
        return Abstract_Exporter::EXPORTER_TYPE_PRODUCTS;
    }

    /**
     * Returns the context phrase for the exporter.
     *
     * @return string The context phrase used for products export.
     */
    protected function get_context_phrase(): string {
        return 'products_export';
    }


    /**
     * Creates a CSV export file for products.
     *
     * Handles the complete CSV export process including setting up filters for
     * delimiter and image columns, processing products in batches, and generating
     * the final CSV file. Supports pagination for large datasets and maintains
     * consistent column headers across export steps.
     *
     * @return string The file path to the generated CSV export file.
     */
    protected function create_csv(): string
    {

        if ( ! defined( 'WC_ABSPATH' ) ) {
            return '';
        }

        ini_set( 'max_execution_time', '28800' );

        add_filter(
            'woocommerce_product_export_delimiter',
            function ( $delimiter ) {
                return ';';
            }
        );

        add_filter('woocommerce_product_object_query',  function ($results) {
            $galleries = array();
            /**
             * @var WC_Product $product
             */
            $unique_images = array();
            $max_columns   = 0;
            foreach ( $results->products as $product ) {
                $gallery        = array();
                $featured_image = get_post_thumbnail_id( $product->get_id() );
                $additional_images = $product->get_gallery_image_ids();
                if ( $product->is_type('variation') ) {
                    $parent_product = wc_get_product( $product->get_parent_id() );
                    if ( $parent_product && ! is_wp_error( $parent_product ) ) {
                        $parent_product_id = $parent_product->get_id();
                        if ( empty( $featured_image ) ) {
                            $featured_image = get_post_thumbnail_id( $parent_product_id );
                        }
                        $additional_images = $parent_product->get_gallery_image_ids();
                    }
                }

                if ( ! empty( $featured_image ) ) {
                    $gallery[] = $featured_image;
                    $gallery   = array_merge( $gallery, $additional_images );
                } else {
                    $gallery = $additional_images;
                }

                if (count($gallery) > $max_columns) {
                    $max_columns = count($gallery);
                }
                if (is_array($gallery) && ! empty($gallery)) {
                    $galleries[$product->get_id()] = $gallery;
                    $unique_images = array_merge( $unique_images, $gallery );
                    $unique_images = array_unique( $unique_images );
                }
            }

            add_filter("woocommerce_product_export_column_names",
                function (array $column_names) use (
                    $unique_images,
                    $galleries,
                    $max_columns
                ) {
                    for ($i = 0; $i <= $max_columns - 1; $i++) {
                        $column_names["image_$i"] = "image_$i";
                        add_filter("woocommerce_product_export_product_column_image_$i",
                            function (
                                $empty_string,
                                WC_Product $product,
                                string $column_id
                            ) use (
                                $unique_images,
                                $i,
                                $galleries
                            ) {

                                $image_link = '';

                                if ( isset( $galleries[$product->get_id()][$i] ) ) {
                                    $image_link = wp_get_attachment_image_url( $galleries[$product->get_id()][$i], 'full' );
                                }

                                if( 0 === $image_link || '0' === $image_link ) {
                                    $image_link = '';
                                }

                                return $image_link;
                            }, 100, 3 );
                    }

                    if ( ! self::$full_columns_headers) {
                        self::$full_columns_headers = $column_names;
                    }

                    if (count($column_names) > count(self::$full_columns_headers)) {
                        self::$full_columns_headers = $column_names;
                    }

                    // we need max length list of column headers to create correct csv header.
                    return self::$full_columns_headers;

                });

            return $results;

        }, 100, 2 );



        try {

            include_once WC_ABSPATH . 'includes/export/class-wc-product-csv-exporter.php';

            $exporter = new Empik_WC_Product_Exporter();
            $exporter->set_product_types_to_export( array( 'simple', 'variable', 'variation' ) );
            $exporter->enable_meta_export( false );
            // $exporter->set_limit( 1 );

            $page                       = isset( $_POST['empik_pr_exp_step'] ) ? absint( $_POST['empik_pr_exp_step'] ) : 1;
            self::$full_columns_headers = isset( $_POST['empik_full_columns'] )
                ? array_map( 'sanitize_text_field', (array) $_POST['empik_full_columns'] )
                : array();

            $this->add_filter_only_checked_products_to_csv();

            $exporter->set_page( $page );
            $exporter->generate_file();

            if ( 100 === $exporter->get_percent_complete() ) {
                if ( method_exists( $exporter, 'get_headers_row_file' ) ) {
                    $csv_data = $exporter->get_headers_row_file() . $exporter->get_file();

                    $product_export_file_name = 'WooCommerce_' . sanitize_file_name(
                            gmdate(
                                'd-m-Y h:i:s',
                                time()
                            ) . '.csv'
                        );
                    update_option( 'empik_last_export_file_name', $product_export_file_name );
                    $new_path = wp_upload_dir()['basedir'] . DIRECTORY_SEPARATOR . $product_export_file_name;

                    file_put_contents(
                        $new_path,
                        $csv_data
                    );

                    try {
                        @unlink(
                            wp_upload_dir()['basedir'] . DIRECTORY_SEPARATOR
                            . $exporter->get_filename()
                        );
                    } catch ( Exception $e ) {

                    }

                    try {
                        @unlink(
                            wp_upload_dir()['basedir'] . DIRECTORY_SEPARATOR
                            . $exporter->get_filename() . '.headers'
                        );
                    } catch ( Exception $e ) {

                    }


                    $this->fix_line_endings( $new_path );

                    return $new_path;
                } else {

                    // legacy support
                    $path = wp_upload_dir()['basedir'] . DIRECTORY_SEPARATOR
                            . $exporter->get_filename();
                    $this->fix_line_endings( $path );

                    return $path;
                }
            } else {

                wp_send_json_success(
                    array(
                        'step'       => ++$page,
                        'percentage' => $exporter->get_percent_complete(),
                        'columns'    => self::$full_columns_headers,
                    )
                );
            }
        } catch ( Exception $e ) {
            Plugin::write_debug_log( $e->getMessage() );
            \wc_get_logger()->debug( 'Błąd eksportu produktów:', array( 'source' => 'empik-log' ) );
            \wc_get_logger()->debug( print_r( $e->getMessage(), true ), array( 'source' => 'empik-log' ) );
            wp_die( 'Błąd eksportu produktów' );
        }

        return '';
    }

    /**
     * Fixes line endings in the CSV file.
     *
     * Removes newline characters from the file content to ensure proper
     * formatting for the export. Replaces the original file with the
     * corrected version.
     *
     * @param string $path The file path to fix line endings for.
     */
    private function fix_line_endings(string $path)
    {
        $str      = file_get_contents($path);
        $newlines = array('\n');
        $fixed    = str_replace($newlines, '', $str);
        unlink($path);
        file_put_contents($path, $fixed);
    }

    private function push_gallery_images_to_separate_columns()
    {

    }


    /**
     * Adds filter to export only checked products to CSV.
     *
     * Conditionally applies a meta query filter to limit the export to only
     * products that have been specifically marked for export, unless the
     * setting is configured to export all products.
     */
    private function add_filter_only_checked_products_to_csv()
    {

        $is_all_products = Plugin::get_settings()->get_option(Settings_Ids::SETTINGS_ID_EXPORT_PRODUCTS_ALL);

        if ('yes' === $is_all_products || ( isset($_POST['empik_is_export_all']) && 'true' === $_POST['empik_is_export_all'] ) ) {

            return [
                'post_type'      => 'product',
                'posts_per_page' => -1,
                'fields'         => 'ids',
            ];

        } else {

            add_filter( 'woocommerce_product_data_store_cpt_get_products_query',
                function ($wp_query_args, $query_vars, $data_store_cpt) {
                    $meta_key = ( new Products_List_Product_import_Column() )->get_meta_id();
                    $meta_value = ( new Products_List_Product_import_Column() )->get_meta_value_true();

                    // Najpierw wykonaj standardowe zapytanie o produkty główne.
                    $parent_args = [
                        'post_type' => 'product',
                        'posts_per_page' => -1,
                        'fields' => 'ids',
                        'post_status' => 'publish',
                        'meta_query' => [
                            [
                                'key' => $meta_key,
                                'value' => $meta_value,
                                'compare' => '=',
                            ]
                        ]
                    ];

                    $parent_query = new WP_Query($parent_args);
                    $parent_ids = $parent_query->posts;
                    wp_reset_postdata();

                    if ( !empty( $parent_ids ) ) {
                        $all_ids = $parent_ids;

                        // Dla każdego produktu sprawdź czy jest zmienny i pobierz warianty.
                        foreach ($parent_ids as $parent_id) {
                            $product = wc_get_product($parent_id);

                            if ($product && $product->is_type('variable')) {
                                $variation_ids = $product->get_children();
                                if (!empty($variation_ids)) {
                                    $all_ids = array_merge($all_ids, $variation_ids);
                                }
                            }
                        }

                        // Usuń oryginalne meta_query i użyj post__in zamiast tego.
                        unset($wp_query_args['meta_query']);
                        $wp_query_args['post__in'] = array_unique($all_ids);
                        $wp_query_args['orderby'] = 'post__in'; // Zachowaj kolejność.
                    } else {
                        // Jeśli nie ma produktów, dodaj meta_query normalnie.
                        $wp_query_args['meta_query'][] = [
                            'key' => $meta_key,
                            'value' => $meta_value,
                            'compare' => '=',
                        ];
                    }

                    return $wp_query_args;

                }, 10, 3 );



        }

    }


    /**
     * Executes the export request to Empik API or test mode.
     *
     * Handles the actual export process by either sending the CSV file to the
     * Empik API or providing test mode functionality with file download links.
     * Updates export status options and cleans up after completion.
     *
     * @param string $path_to_csv The file path to the CSV export file.
     *
     * @return Ajax_Response_Model The response model containing export results.
     */
    protected function do_request(string $path_to_csv): Ajax_Response_Model
    {

        update_option('empik_export_products_is_running', true, true);

        if ( defined('EMPIK_TEST_MODE_ON') && true === EMPIK_TEST_MODE_ON ) {

            //for testing - generate .csv file but do no export do Empik dashboard
            if ( ! empty(get_option('empik_last_export_file_name'))) {

                $uploads               = wp_upload_dir();
                $last_export_file_url  = esc_url($uploads['baseurl'] . '/' . get_option('empik_last_export_file_name'));
                $last_export_file_name = sanitize_text_field(get_option('empik_last_export_file_name'));

                delete_option('empik_last_export_file_name');

                delete_option('empik_export_products_is_running');

                return (new Ajax_Response_Model(Ajax_Response_Model::SUCCESS,
                    sprintf("%s %s <br>%s <a download href='%s'>%s</a>",
                        esc_html__('Export finished. ID:', 'empik-for-woocommerce'),
                        "TEST products export",
                        esc_html__('File:', 'empik-for-woocommerce'),
                        $last_export_file_url,
                        $last_export_file_name
                    )));


            } else {

                delete_option('empik_export_products_is_running');

                return (new Ajax_Response_Model(Ajax_Response_Model::SUCCESS,
                    sprintf(esc_html__("Export finished. ID: %s ", 'empik-for-woocommerce'),
                        "TEST products export")));
            }


        } else {

            $result = $this->api->export_products($path_to_csv);

            delete_option('empik_export_products_is_running');

            return $result;
        }


    }


    /**
     * Retrieves all unique product attributes from the database.
     *
     * Queries the database for all product attribute meta values, unserializes them,
     * and extracts unique attribute keys. Returns a sorted array of attribute names
     * or taxonomy slugs.
     *
     * @param string $post_type The post type to query for attributes. Default 'product'.
     *
     * @return array Sorted array of unique attribute keys/names.
     */
    public function get_all_product_attributes($post_type = 'product')
    {
        global $wpdb;

        $results = $wpdb->get_col($wpdb->prepare(
            "SELECT DISTINCT pm.meta_value
            FROM {$wpdb->postmeta} AS pm
            LEFT JOIN {$wpdb->posts} AS p ON p.ID = pm.post_id
            WHERE p.post_type = %s
            AND p.post_status IN ( 'publish', 'pending', 'private', 'draft' )
            AND pm.meta_key = '_product_attributes'",
            $post_type
        ));

        // Go through each result, and look at the attribute keys within them.
        $result = array();

        if ( ! empty($results)) {
            foreach ($results as $_product_attributes) {
                $attributes = maybe_unserialize(maybe_unserialize($_product_attributes));
                if ( ! empty($attributes) && is_array($attributes)) {
                    foreach ($attributes as $key => $attribute) {
                        if ( ! $key) {
                            continue;
                        }
                        if ( ! strstr($key, 'pa_')) {
                            if (empty($attribute['name'])) {
                                continue;
                            }
                            $key = $attribute['name'];
                        }

                        $result[$key] = $key;
                    }
                }
            }
        }

        sort($result);

        return $result;
    }


}
